#!/bin/bash

TARGET_DIR=/usr/share/biosfw
EXE_FILE=H2OFFT-Lx64

BOOT_CNTRL_REG=0x0B
CS_SWI_BIT_MASK=0xCF
BIT0=$((0x1 << 0))  #CPU booted bit
BIT1=$((0x1 << 1))  #linux up bit
BIT2=$((0x1 << 2))  #reserved
BIT3=$((0x1 << 3))  #FLASH image, booted from CS0 or CS1
BIT4=$((0x1 << 4))  #FLASH select, which flash to select
BIT5=$((0x1 << 5))  #flash_swizzle_en 1-auto 0-manual
BIT6=$((0x1 << 6))  #watchodg enable
BIT7=$((0x1 << 7))  #watchdog interrrupt enable


ORIG_FLASH=$(re_fpga -r $BOOT_CNTRL_REG | awk '{print $8}')
cd ${TARGET_DIR}

function cleanup()
{
    CURR_FLASH=$(re_fpga -r $BOOT_CNTRL_REG | awk '{print $8}')

    if [ "$ORIG_FLASH" != "$CURR_FLASH" ]; then
            # Reset it back to avoid funny behavior
            # Set all the bits to except swizzle bit 
            DATA=$(($ORIG_FLASH & ~$BIT5))
            printf -v W_DATA "%x" "$DATA"
            re_fpga -w $BOOT_CNTRL_REG $W_DATA > /dev/null 2>&1 
           
            # restore the swizzle bit
            DATA=$(re_fpga -r $BOOT_CNTRL_REG | awk '{print $8}')
            DATA=$(($ORIG_FLASH | $BIT5))
            printf -v W_DATA "%x" "$DATA"
            re_fpga -w $BOOT_CNTRL_REG $W_DATA > /dev/null 2>&1	    
            ./${EXE_FILE} -DR > /dev/null 2>&1
    fi
}

trap cleanup EXIT

function print_version() 
{
    # Re-Enable driver in Linux using '-DR' after selecting flash
    ./${EXE_FILE} -DR > /dev/null 2>&1
    ./${EXE_FILE} -SC

}

function print_long_version ()
{
    print_version | grep Version  | awk '{print $5}'
}

function print_short_version ()
{
    print_long_version | awk -F. '{print $2 "." $3 "." $4 "." $5}'
}

function select_flash_a()
{
    # Select flash A
    # Set flash swizzle to manual and chip select to zero 
    DATA=$((ORIG_FLASH & $CS_SWI_BIT_MASK))

    printf -v W_DATA "%x" "$DATA"
    # write the data to  chip select
    re_fpga -w $BOOT_CNTRL_REG $W_DATA > /dev/null 2>&1

}

function select_flash_b()
{
    # Select flash B
    # Set flash swizzle to manual and chip select to 1
    DATA=$((ORIG_FLASH & $CS_SWI_BIT_MASK))
    DATA=$((DATA | $BIT4))	

    printf -v W_DATA "%x" "$DATA"
    # write the data to  chip select
    re_fpga -w $BOOT_CNTRL_REG $W_DATA > /dev/null 2>&1
}

function print_selected_flash() 
{
    # Check which Flash is selected
    DATA=$(re_fpga -r $BOOT_CNTRL_REG | awk '{print $8}')
    CS=$(($DATA >> 0x04))
    CS=$(($CS & 0x01))
    if [ 1 == "$CS" ]; then
	echo "FLASH B:"
    else
	echo "FLASH A:"
    fi
}

function main ()
{
    FLASH="X"
    while getopts ":aAbBlLsS" OPT; do
        case "$OPT" in
            a|A)
                FLASH="A"
                ;;
            b|B)
                FLASH="B"
                ;;
            l|L)
                VERSION="LONG"
                ;;
            s|S)
                VERSION="SHORT"
                ;;
        esac
    done

    if [ "$FLASH" == "X" ]; then
        select_flash_a
        print_selected_flash
        print_version

        select_flash_b
        print_selected_flash
        print_version
        return
    elif [ "$FLASH" == "A" ]; then
        select_flash_a
    elif [ "$FLASH" == "B" ]; then
        select_flash_b
    fi
    if [ "$VERSION" == "LONG" ]; then
        print_long_version
    elif [ "$VERSION" == "SHORT" ]; then
        print_short_version
    else
        print_selected_flash
        print_version
    fi
}

# Entry point
main $@

exit $?
