#!/bin/bash

TARGET_DIR=/usr/share/biosfw
EXE_FILE=H2OFFT-Lx64
BIOS_FILE=
FLASHROM=/usr/sbin/flashrom
FOUND_CHIP="NONE"

declare -a NGSPMB_CHIPS

# N25Q064A13ESFD0F  MFR_ID:0x20 DEV_ID:0xBA17
NGSPMB_CHIPS[0]="N25Q064..3E"
# MX25L6433FMI-08G  MFR_ID:0xC2 DEV_ID:0x2017
NGSPMB_CHIPS[1]="MX25L6436E/MX25L6445E/MX25L6465E/MX25L6473E"
# M25P64  MFR_ID: 0x20 DEV_ID:0x2017
NGSPMB_CHIPS[2]="M25P64"

ORIG_FLASH=$(cat  /sys/devices/platform/ngpmb-cpld/flash_select)
cd ${TARGET_DIR}

function cleanup()
{
    CURR_FLASH=$(cat  /sys/devices/platform/ngpmb-cpld/flash_select)

    if [ "$ORIG_FLASH" != "$CURR_FLASH" ]; then
        # Reset it back to avoid funny behavior
        echo $ORIG_FLASH > /sys/devices/platform/ngpmb-cpld/flash_select
    fi

    # Restore flash swizzle only after the flash select has been restored
    CURR_FLASH_SWIZZLE_EN=$(cat /sys/devices/platform/ngpmb-cpld/flash_swizzle_enable)
    if [ "$ORIG_FLASH_SWIZZLE_EN" != "$CURR_FLASH_SWIZZLE_EN" ]; then
        echo $ORIG_FLASH_SWIZZLE_EN > /sys/devices/platform/ngpmb-cpld/flash_swizzle_enable
    fi
}

trap cleanup EXIT

function probe_flashrom_support_chip()
{
    for chip in ${NGSPMB_CHIPS[@]}; do
        ${FLASHROM} -p internal -c ${chip}
        if [ $? -eq 0 ]; then
            FOUND_CHIP=${chip}
            return 0
        fi
    done
    FOUND_CHIP="NONE"
    return 1
}

function upgrade_bios()
{
    probe_flashrom_support_chip
    if [ $? -eq 0 ]; then
        ${FLASHROM} -p internal -c ${FOUND_CHIP} -w ${BIOS_FILE}
        ret=$?
        if [ $? -eq 0 ]; then
            echo "Flash complete!"
        fi
    else
        echo "Chip not supported for upgrade"
        return 1
    fi
    return $ret
}

#
# Hack for non SPMB boards
# Boards other than spmb have a bug in the bootcpld. When a flash swizzle
# occurs, programming the flash_select to the non current flash doesn't take
# effect unless the flash_swizzle is disabled
function check_flash_swizzle()
{
    case "${BOARD}" in
        7)
            return
            ;;
        *)
            if [ "$FLASH_SWIZZLE" = "1" ]; then
                echo 0 > /sys/devices/platform/ngpmb-cpld/flash_swizzle_enable
            fi
            ;;
    esac
}

function select_flash_a()
{
    # Select flash A
    echo A > /sys/devices/platform/ngpmb-cpld/flash_select
}

function select_flash_b()
{
    # Select flash B
    echo B > /sys/devices/platform/ngpmb-cpld/flash_select
}

function print_selected_flash()
{
    echo "Upgrading Flash `cat  /sys/devices/platform/ngpmb-cpld/flash_select`:"
}

function upgrade_stout_pmb ()
{
    FLASH="X"
    while getopts ":aAbBf:" OPT; do
        case "$OPT" in
            a|A)
                FLASH="A"
                ;;
            b|B)
                FLASH="B"
                ;;
            f)
                BIOS_FILE=$OPTARG
        esac
    done

    if [ -z "${BIOS_FILE}" ]; then
        echo "Usage: Missing -f <bios file>"
        return 1
    elif [ ! -f "${BIOS_FILE}" ]; then
        echo "Usage: file $BIOS_FILE not found"
        return 2
    fi

    BOARD=$(cat /sys/jnx/card/local/type)
    FLASH_SWIZZLE=$(cat /sys/devices/platform/ngpmb-cpld/flash_swizzle)
    ORIG_FLASH_SWIZZLE_EN=$(cat /sys/devices/platform/ngpmb-cpld/flash_swizzle_enable)

    if [ "$FLASH" == "X" ]; then
        echo "Usage: Missing -a/A -b/B"
        return 3
    elif [ "$FLASH" == "A" ]; then
        check_flash_swizzle
        select_flash_a
    elif [ "$FLASH" == "B" ]; then
        check_flash_swizzle
        select_flash_b
    fi

    print_selected_flash
    upgrade_bios
    return $?
}

# Entry point
bios_version=$(dmidecode -s bios-version | grep -v "^#" | grep ^[A-Z]*\_[A-Z]*\_[A-Z0-9]*[0-9]\_ | sed s/^[A-Z]*\_[A-Z]*\_[A-Z0-9]*[0-9]\_//)
if [ -z "$bios_version" ]; then
    upgrade_stout_pmb $@
else
    # Common BIOS will upgrade the BIOS over pxe. We just need to set the appropriate flag.
    /usr/sbin/OsIndicate.sh -s FILE_CAPSULE_DELIVERY
    ret=$?
    if [ "$ret" -eq 0 ]; then
        echo "Flash staged!"
    else
        echo "Error running OsIndicate.sh script"     
    fi
fi
exit $?
